'use strict'

entityRegistry['module']['threeDeeBalls'] = {
    extendedInfo: {
        displayName: 'Balls',
        displayGroup: '3D Effects',
    },
    init: () => {
        const makeIoshedronModel = (subDivisions) => {
            const X = .525731112119133606
            const Z = .850650808352039932
            const N = 0
            let vertices = [
                [-X,N,Z], [X,N,Z], [-X,N,-Z], [X,N,-Z],
                [N,Z,X], [N,Z,-X], [N,-Z,X], [N,-Z,-X],
                [Z,X,N], [-Z,X, N], [Z,-X,N], [-Z,-X, N]
            ].map(p => m4.multiplyVector(p, 1))
            let triangles = [
                [0,4,1],[0,9,4],[9,5,4],[4,5,8],[4,8,1],
                [8,10,1],[8,3,10],[5,3,8],[5,2,3],[2,7,3],
                [7,10,3],[7,6,10],[7,11,6],[11,0,6],[0,1,6],
                [6,1,10],[9,0,11],[9,11,2],[9,2,5],[7,2,11],
            ]
            const findOrAdd = (p) => {
                const i = vertices.findIndex(f => f[0]===p[0] && f[1]===p[1] && f[2]===p[2])
                if (i !== -1) return i
                return vertices.push(p)-1
            }
            for (let i = 0; i < subDivisions; ++i) {
                const numTriangles = triangles.length
                const newTriangles = []
                for (let i = 0; i < numTriangles; ++i) {
                    const triangle = triangles[i]
                    const v0 = triangle[0]
                    const v1 = triangle[1]
                    const v2 = triangle[2]
                    const p0 = vertices[v0]
                    const p1 = vertices[v1]
                    const p2 = vertices[v2]
                    const p01 = m4.normalize([(p0[0]+p1[0])/2, (p0[1]+p1[1])/2, (p0[2]+p1[2])/2])
                    const p12 = m4.normalize([(p1[0]+p2[0])/2, (p1[1]+p2[1])/2, (p1[2]+p2[2])/2])
                    const p20 = m4.normalize([(p2[0]+p0[0])/2, (p2[1]+p0[1])/2, (p2[2]+p0[2])/2])
                    const v01 = findOrAdd(p01)
                    const v12 = findOrAdd(p12)
                    const v20 = findOrAdd(p20)
                    newTriangles.push([v20, v0, v01])
                    newTriangles.push([v01, v1, v12])
                    newTriangles.push([v12, v2, v20])
                    newTriangles.push([v20, v01, v12])
                }
                triangles = newTriangles
            }

            let lines = [] // dedupe!
            const lineAdd = (line) => {
                // if (!lines.find((x => x[0] === line[0] && line[1] === line[1]))) {
                    lines.push(line)
                // }
            }
            triangles.forEach(triangle => {
                lineAdd([triangle[0], triangle[1]])
                lineAdd([triangle[1], triangle[2]])
                lineAdd([triangle[2], triangle[0]])
            })

            return {
                verts: vertices,
                objects: [
                    {
                        name: 'bob',
                        subObjects: [
                            {
                                diffuse: [1, 1 ,1],
                                ambient: [0, 0, 0],
                                emissive: [0, 0, 0],
                                lines: lines,
                                tris: triangles,
                                quads: [],
                            },
                        ]
                    }
                ]
            }
        }

        return {
            models: [
                makeIoshedronModel(0),
                makeIoshedronModel(1),
                makeIoshedronModel(2),
                makeIoshedronModel(3),
                makeIoshedronModel(4),
            ]
        }
    },
    staticConfig: [
    ],
    dynamicConfig: [
        { paramName: 'position', displayName: 'Position', type: 'float3', defaultValue: [0, 0, 0]},
        { paramName: 'rotation', displayName: 'Rotation', type: 'angle3', defaultValue: [0, 0, 0]},
        { paramName: 'scale', displayName: 'Scale', type: 'float3', defaultValue: [1, 1, 1]},
        { paramName: 'diffuse', displayName: 'Diffuse', type: 'color', defaultValue: [1, 1, 1]},
        { paramName: 'emissive', displayName: 'Emissive', type: 'color', defaultValue: [0, 0, 0]},
        { paramName: 'subdivisions', displayName: 'Subdivisions', type: 'int', defaultValue: 0},
        { paramName: 'noisePosition', displayName: 'Noise Position', type: 'float3', defaultValue: [0, 0, 0]},
        { paramName: 'noiseScale', displayName: 'Noise Scale', type: 'float', defaultValue: .75},
        { paramName: 'noiseDistance', displayName: 'Noise Distance', type: 'float', defaultValue: .6},
        { paramName: 'drawMode', displayName: 'Draw Mode', type: 'string', defaultValue: 'tris', uiOptions: { options: [{text:'Triangles',value:'tris'}, {text:'Lines',value:'lines'}, {text:'Both',value:'both'}]}},
    ],
    actions: {
        'render': (self, frameTime, config, ctx) => {
            const {
                position,
                rotation,
                scale,
                diffuse,
                emissive,
                subdivisions,
                noisePosition,
                noiseScale,
                noiseDistance,
                drawMode,
            } = { ...config }

            const colorBuffer = renderer.getCurrentBuffer('color')
            const depthBuffer = renderer.getCurrentBuffer('depth')
            const brightnessBuffer = renderer.getCurrentBuffer('brightness')

            const xRotationMat = m4.xRotation(rotation[0])
            const yRotationMat = m4.yRotation(rotation[1])
            const zRotationMat = m4.zRotation(rotation[2])
            const scaleMat = m4.scaling(scale[0], scale[1], scale[2])
            const translationMat = m4.translation(position[0], position[1], position[2])
            const modelMat = m4.multiply(translationMat, m4.multiply(scaleMat, m4.multiply(m4.multiply(zRotationMat, yRotationMat), xRotationMat)))
            
            const trans = m4.translation(0, 0, 0)
            const worldMat = m4.multiply(trans, modelMat)

            const model = self.models[subdivisions]
            const oldVerts = model.verts
            model.verts = model.verts.map(p => {
                const offset = noise.simplex3(noisePosition[0]+p[0]*noiseScale,noisePosition[1]+p[1]*noiseScale,noisePosition[2]+p[2]*noiseScale)
                return m4.multiplyVector(p, -(1+offset*noiseDistance))
            })
            model.objects[0].subObjects[0].diffuse = diffuse
            model.objects[0].subObjects[0].emissive = emissive
            renderer.decorateModel(model)
            if (drawMode === 'both') {
                renderer.drawModel(model, worldMat, colorBuffer, depthBuffer, brightnessBuffer)
            } else if (drawMode === 'tris') {
                const save = model.objects[0].subObjects[0].lines
                model.objects[0].subObjects[0].lines = []
                renderer.drawModel(model, worldMat, colorBuffer, depthBuffer, brightnessBuffer)
                model.objects[0].subObjects[0].lines = save
            } else if (drawMode === 'lines') {
                const save = model.objects[0].subObjects[0].tris
                model.objects[0].subObjects[0].tris = []
                renderer.drawModel(model, worldMat, colorBuffer, depthBuffer, brightnessBuffer)
                model.objects[0].subObjects[0].tris = save
            }
            model.verts = oldVerts
        }
    }
}
